using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Text;
using System.Windows.Forms;
using System.ComponentModel;

namespace DemoApplication
{
	/// <summary>
	/// 
	/// </summary>
	public class MyCustomControl : Control
	{
		private readonly Color _DefaultShadowColor = SystemColors.ControlDark;
		private Color m_objShadowColor;
		private int m_nShadowOffset = 2;

		private void DrawText( PaintEventArgs e, Rectangle textRectangle, Color textColor )
		{
			TextRenderer.DrawText(
				e.Graphics,
				this.Text,
				this.Font,
				textRectangle,
				textColor,
				TextFormatFlags.SingleLine );
		}

		/// <summary>
		/// Raises the <see cref="E:System.Windows.Forms.Control.Paint"></see> event.
		/// </summary>
		/// <param name="e">A <see cref="T:System.Windows.Forms.PaintEventArgs"></see> that contains the event data.</param>
		protected override void OnPaint( PaintEventArgs e )
		{
			Rectangle rtText = this.ClientRectangle;
			rtText.X += this.Padding.Left;
			rtText.Width -= this.Padding.Horizontal;
			rtText.Y += this.Padding.Top;
			rtText.Height -= this.Padding.Vertical;

			GroupBoxRenderer.DrawGroupBox( e.Graphics, this.ClientRectangle, System.Windows.Forms.VisualStyles.GroupBoxState.Normal );

			if (m_nShadowOffset > 0)
			{
				rtText.Offset( m_nShadowOffset, m_nShadowOffset );
				DrawText( e, rtText, this.ShadowColor );

				rtText.Offset( -m_nShadowOffset, -m_nShadowOffset );
				DrawText( e, rtText, this.ForeColor );
			}
			else
			{
				DrawText( e, rtText, this.ShadowColor );
				int nShadowOffset = Math.Abs( m_nShadowOffset );

				rtText.Offset( nShadowOffset, nShadowOffset );
				DrawText( e, rtText, this.ForeColor );
			}

			if (this.Focused)
			{
				Rectangle rtFocus = rtText;
				rtFocus.Inflate( -3, -3 );
				ControlPaint.DrawFocusRectangle( e.Graphics, rtFocus );
			}
		}

		/// <summary>
		/// Raises the <see cref="E:System.Windows.Forms.Control.Enter"></see> event.
		/// </summary>
		/// <param name="e">An <see cref="T:System.EventArgs"></see> that contains the event data.</param>
		protected override void OnEnter( EventArgs e )
		{
			base.OnEnter( e );
			this.Invalidate();
		}

		/// <summary>
		/// Raises the <see cref="E:System.Windows.Forms.Control.Leave"></see> event.
		/// </summary>
		/// <param name="e">An <see cref="T:System.EventArgs"></see> that contains the event data.</param>
		protected override void OnLeave( EventArgs e )
		{
			base.OnLeave( e );
			this.Invalidate();
		}
		//===================================================
		// Public
		//===================================================

		/// <summary>
		/// Gets or sets a value indicating whether the control is automatically resized to display its entire contents.
		/// </summary>
		/// <value></value>
		/// <returns>true if the control adjusts its width to closely fit its contents; otherwise, false. The default is true.</returns>
		[DefaultValue( true )]
		[Browsable( true )]
		[EditorBrowsable( EditorBrowsableState.Always )]
		public override bool AutoSize
		{
			get { return base.AutoSize; }
			set { base.AutoSize = value; }
		}

		/// <summary>
		/// Retrieves the size of a rectangular area into which a control can be fitted.
		/// </summary>
		/// <param name="proposedSize">The custom-sized area for a control.</param>
		/// <returns>
		/// An ordered pair of type <see cref="T:System.Drawing.Size"></see> representing the width and height of a rectangle.
		/// </returns>
		public override Size GetPreferredSize( Size proposedSize )
		{
			Size objTextSize = TextRenderer.MeasureText(
				this.Text,
				this.Font,
				proposedSize,
				TextFormatFlags.SingleLine );

			Size objControlSize = new Size(
				objTextSize.Width + this.Padding.Horizontal + Math.Abs( m_nShadowOffset ),
				objTextSize.Height + this.Padding.Vertical + Math.Abs( m_nShadowOffset ) );

			return objControlSize;
		}

		#region ... ShadowColor property ...
		/// <summary>
		/// Gets or sets the color of the shadow.
		/// </summary>
		/// <value>The color of the shadow.</value>
		[Category( "Appearance" )]
		[Description( "The color of the shadow of the text." )]
		public Color ShadowColor
		{
			get { return m_objShadowColor; }
			set
			{
				if (m_objShadowColor == value)
					return;

				m_objShadowColor = value;
				this.Invalidate();
			}
		}

		/// <summary>
		/// Returns a value indicating whether the value of the <see cref="ShadowColor"/> property should be serialized.
		/// </summary>
		/// <returns></returns>
		public bool ShouldSerializeShadowColor()
		{
			return (!Color.Equals( m_objShadowColor, _DefaultShadowColor ));
		}

		/// <summary>
		/// Resets the value of the <see cref="ShadowColor"/> property.
		/// </summary>
		/// <returns></returns>
		public void ResetShadowColor()
		{
			this.ShadowColor = _DefaultShadowColor;
		}
		#endregion


		/// <summary>
		/// Gets or sets the horizontal and vertical offset of the shadow in relation to the text (in pixels).
		/// </summary>
		/// <value>The shadow offset.</value>
		[DefaultValue( 2 )]
		[Category( "Appearance" )]
		[Description( "The horizontal and vertical offset of the shadow in relation to the text (in pixels)" )]
		public int ShadowOffset
		{
			get { return m_nShadowOffset; }
			set
			{
				if (m_nShadowOffset == value)
					return;
				m_nShadowOffset = value;
				this.Invalidate();
			}
		}

		/// <summary>
		/// Initializes a new instance of the <see cref="MyCustomControl"/> class.
		/// </summary>
		public MyCustomControl()
		{
			base.AutoSize = true;
			this.SetAutoSizeMode( AutoSizeMode.GrowAndShrink );
			this.SetStyle( ControlStyles.ResizeRedraw, true );

			m_objShadowColor = _DefaultShadowColor;
		}
	}
}
