using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;

namespace SOAApplication.Services
{
	interface IVisitedElement
	{
		int NoOfVisits { get;}
		XmlElement Element { get;}
	}

	delegate void TopChangedEvent(IVisitedElement newTop, IVisitedElement oldTop);
	delegate void MOUListChangedEvent(IMOUService source);

	interface IMOUService
	{
		IVisitedElement Top { get;}
		IVisitedElement[] MOUList { get;}
		event TopChangedEvent OnTopChanged;
		event MOUListChangedEvent OnMOUListChanged;

		void Reset();
	}

	class MOUService : IMOUService, IService
	{
		class VisitedElement : IVisitedElement
		{

			#region constructors

			public VisitedElement(XmlElement element)
			{
				m_element = element;
			}

			#endregion

			#region private state

			private int m_noOfVisists = 1;
			private XmlElement m_element;

			#endregion

			#region IVisitedElement members

			public int NoOfVisits
			{
				get { return m_noOfVisists; }
				set { m_noOfVisists = value; }
			}

			public XmlElement Element
			{
				get { return m_element; }
				set { m_element = value; }
			}

			#endregion

			#region output

			public override string ToString()
			{
				return String.Format("{0} ({1})", m_element.Name, NoOfVisits);
			}

			#endregion
		}

		#region private state

		List<VisitedElement> m_visitedElements = new List<VisitedElement>();

		#endregion

		#region core implementation

		void ProcessVisit(XmlElement element)
		{
			int idx = 0;
			while ((idx < m_visitedElements.Count) && (m_visitedElements[idx].Element != element)) idx++;

			if (idx < m_visitedElements.Count)
			{
				//element alreay in list
				m_visitedElements[idx].NoOfVisits++;
			}
			else
			{
				//first time visited
				m_visitedElements.Add(new VisitedElement(element));
			}

			//move updated element to front
			VisitedElement changedElement = m_visitedElements[idx];
			int targetIndex = idx;
			while ((targetIndex > 0) && (m_visitedElements[targetIndex - 1].NoOfVisits <= changedElement.NoOfVisits)) targetIndex--;

			if (targetIndex < idx)
			{
				m_visitedElements.RemoveAt(idx);
				m_visitedElements.Insert(targetIndex, changedElement);
            }

            if ((targetIndex == 0) && ((targetIndex < idx) || (m_visitedElements.Count == 1))  && (OnTopChanged != null)) OnTopChanged(changedElement, (m_visitedElements.Count > 1) ? m_visitedElements[1] : null);

			if (OnMOUListChanged != null) OnMOUListChanged(this);
		}

		#endregion

		#region IMOUService Members

		public IVisitedElement Top
		{
			get { return (m_visitedElements.Count > 0) ? m_visitedElements[0] : null; }
		}

		public IVisitedElement[] MOUList
		{
			get 
			{  
				IVisitedElement[] result = new IVisitedElement[ m_visitedElements.Count];
				for (int idx = 0; idx < m_visitedElements.Count; idx++)
				{
					result[idx] = m_visitedElements[idx];
				}
				return result;
			}
		}

		public event TopChangedEvent OnTopChanged;

		public event MOUListChangedEvent OnMOUListChanged;

		public void Reset()
		{
			IVisitedElement oldTop = Top;
			m_visitedElements.Clear();
			if ((OnTopChanged != null) && (oldTop != null)) OnTopChanged(null, oldTop);
			if (OnMOUListChanged != null) OnMOUListChanged(this);
		}

		#endregion

		#region IService Members

		public void OnServiceAvailable(Type serviceType)
		{
			if (serviceType.Equals(typeof(ISelectionService)))
			{
				ServiceBroker.GetService<ISelectionService>().OnSelectionChanged += new SelectionChangedEvent(MOUService_OnSelectionChanged);
				return;
			}

			if (serviceType.Equals(typeof(IModelService)))
			{
				ServiceBroker.GetService<IModelService>().OnModelChanged += new ModelChangedEvent(MOUService_OnModelChanged);
			}
		}

		#endregion

		#region service dependencies

		void MOUService_OnModelChanged()
		{
			Reset();
		}

		void MOUService_OnSelectionChanged(XmlElement newSelection)
		{
			if (newSelection != null) ProcessVisit(newSelection);
		}

		#endregion
	}
}
